/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Collections.Specialized;
using System.Collections;
using System.Runtime.Serialization;
using Borland.Eco.Exceptions;
using Borland.Eco.DataRepresentation;
using Borland.Eco.Persistence;
using Borland.Eco.Persistence.ORMapping;
using Borland.Eco.Persistence.Connection;
using Borland.Eco.Persistence.Configuration;

namespace Borland.Eco.Persistence
{
	public enum IdAllocationMode
	{
		FromAttributes,
		PreUpdate,
		PostUpdate
	}

	[Serializable]
	public class EcoCantGetNewObjectId: EcoException
	{
		public EcoCantGetNewObjectId(): base() {}
		public EcoCantGetNewObjectId(string message): base(message) {}
		public EcoCantGetNewObjectId(string message, Exception innerException): base(message, innerException) {}
	}

	///<summary>
	/// This interface is used by the Eco persistence mapper to handle different strategies of dealing with database keys.
	/// <br/>
	/// there are four different prewritten implementations of this interface that handles the default eco id mechanism,
	/// autoinc, keys mapped to attributes and finally keys that are guids.
	///</summary>
	public interface IKeyMapper
	{
		///<summary>
		///This method fills the parameter objects with data from an id
		///</summary>
		void IdToParameters(ObjectId id, IDataParameter[] parameters);
		///<summary>
		///This method creates a new id based on the values of the field objects
		///</summary>
		ObjectId IdFromFields(IField[] keyFields, ArrayList keyMappers, int classId, int timeStamp);
		///<summary>
		///Returns the number of columns that are used to store the id
		///</summary>
		int ColumnCount();
		///<summary>
		///This method is called once for each object before objects are updated (if IdAllocationMode = PreUpdate and allows the keymapper to prepare for the number of IDs that needs to be allocated
		///</summary>
		void ReserveId();
		///<summary>
		///This method is called once for each object (if IdAllocationMode = PreUpdate) that needs to have a persistent id. It must return a new id with the same ClassId as the old id
		///</summary>
		ObjectId NewIdPreUpdate(IDatabase db, ObjectId oldId);
		///<summary>
		///This method is called once for each object before it is updated (if IdAllocationMode = FromAttributes).
		///<param name="objContents">The object that needs an id</param>
		///<param name="memberIndexes">An arraylist of MemberIndexColumnIndex objects</param>
		///<param name="translationList">If the new Id contains any information from relations to other objects, it is important that those ids are translated using the translationlist before the related object ids are used</param>
		///</summary>
		ObjectId NewIdFromAttributes(ObjectContents objContents, ArrayList memberIndexes, IdTranslationList translationList);
		///<summary>
		///This method can add additional tables/columns if the keymapper requires any additional
		///structure in the database schema. It is called once per session
		///</summary>
		void InitializePSDescriptions(DatabaseDefinition db);
		///<summary>
		///This method may perform initialization of the database such as installing generators for autoinc mechanisms
		///</summary>
		void InitializeDatabase(IDatabase db);
		///<summary>
		///Same as InitializeDatabase, but rather than operating directly on the database, the initializations should be returned as sql statements to be used in a script
		///</summary>
		void InitializeDatabaseScript(StringCollection script);
		///<summary>
		///Indicates the strategy used by the key mapper.
		///FromAttributes means that the key mapper will extract the key values from the contents of the object.
		///PreUpdate means that the id values can be allocated prior to updating the object (used by the default eco key mapper and the guid-mapper)
		///PostUpdate means that the id values can only be obtained after the object has been inserted to the database (used by autoinc mapper)
		///</summary>
		IdAllocationMode GetIdAllocationMode();
		///<summary>
		///Returns the sql type for the respective column
		///</summary>
		string ColumnType(ArrayList KeyMappers, int columnNum);
		///<summary>
		///This method is called after the key mapper has been created to initialize the keymapper with some basic configuration
		///</summary>
		void Initialize(SqlDatabaseConfig config, ArrayList keyMappers);
		///<summary>
		///Returns a Where clause condition that specifies a condition that is only met by the objects in the idlist
		///</summary>
		string IdListToWhereFragment(ObjectIdCollection idList, IParameterized parameterized, StringCollection idColumns);
		///<summary>
		///Returns a Where clause condition that specifies a condition that is only met by the objects with the specified id.
		///Example: ECO_ID in (1, 5, 9)
		///Example: ECO_ID = 2
		///Example: ECO_ID in (?, ?, ?, ?)  [actual datavalues are stored in parameters]
		///Example composite key: (FIRSTNAME='Peter' AND LASTNAME='Smith') OR (FIRSTNAME='John' AND LASTNAME='Lane') OR (FIRSTNAME='Dave' AND LASTNAME='Short')
		///</summary>
		string IdToWhereFragment(ObjectId id, IParameterized parameterized, StringCollection idColumns);
	}
}
